<?php

include_once "log.php";
include_once "globals.php";

set_time_limit(0);
date_default_timezone_set('Europe/Berlin');

/* ----------------------------------------------------------------- */
/*                       POST/GET-Parameter verarbeiten              */
// Priorität für POST, Fallback auf GET
// GET-Parameter verarbeiten
$startOffset = isset($_GET['start']) ? (int)$_GET['start'] : 0;
$endOffset = isset($_GET['end']) ? (int)$_GET['end'] : 1;

/* ----------------------------------------------------------------- */
/*                              RangeTime                            */

$timeZone = 'Europe/Berlin';
$date = new DateTime('today', new DateTimeZone($timeZone));
$now = new DateTime('now', new DateTimeZone($timeZone));

// Datenbank Delete Time
$datenbank_vorgestern = clone $date;
$datenbank_vorgestern->sub(new DateInterval("P2D"))->setTime(0, 0, 0);
$datenbank_vorgestern = $datenbank_vorgestern->format('Y-m-d H:i:s');

$datenbank_vor_10_tagen = clone $date;
$datenbank_vor_10_tagen->sub(new DateInterval("P10D"))->setTime(0, 0, 0);
$datenbank_vor_10_tagen = $datenbank_vor_10_tagen->format('Y-m-d H:i:s');

// Datum berechnen
$dateRange = calculateDateRangeWithYesterday($date, $startOffset, $endOffset);


$rangeStartYesterday = $dateRange['yesterdayDate'];
$rangeStart = $dateRange['startDate'];
$rangeEnd = $dateRange['endDate'];

$rangeStartYesterdayMitZeit = $dateRange['yesterdayDateTime'];
$rangeStartMitZeit = $dateRange['startDateTime'];
$rangeEndMitZeit = $dateRange['endDateTime'];

// Shutdown-Funktion für Fehlerbehandlung
register_shutdown_function(function () {
    $error = error_get_last();
    if ($error) {
        echo "<pre>";
        echo "Das Skript wurde unterbrochen. Letzter Fehler:\n";
        print_r($error);
        echo "</pre>";
    } else {
        echo "Das Skript wurde erfolgreich ausgeführt oder beendet.";
    }
    exit;
});

/* ----------------------------------------------------------------- */
/*                          Beispielverwendung                       */
echo "rangeStart\n";
echo "<br>";
echo $rangeStart;
echo "<br>";
echo "rangeEnd\n";
echo "<br>";
echo $rangeEnd;
echo "<br>";
echo "rangeStartMitZeit\n";
echo "<br>";
echo $rangeStartMitZeit;
echo "<br>";
echo "rangeEndMitZeit\n";
echo "<br>";
echo $rangeEndMitZeit;
echo "<br>";
echo "rangeStartYesterday\n";
echo "<br>";
echo $rangeStartYesterday;
echo "<br>";
echo "rangeStartYesterdayMitZeit\n";
echo "<br>";
echo $rangeStartYesterdayMitZeit;
echo "<br>";
echo "datenbank_vor_10_tagen\n";
echo "<br>";
echo $datenbank_vor_10_tagen;
echo "<br>";


// Zeitzonen-Informationen
echo 'Globale Standardzeitzone: ' . date_default_timezone_get() . "<br>";
echo 'Einstellung in php.ini: ' . ini_get('date.timezone') . "<br>";

if (date_default_timezone_get()) {
    echo 'date_default_timezone_set: ' . date_default_timezone_get() . '<br />';
}

if (ini_get('date.timezone')) {
    echo 'date.timezone: ' . ini_get('date.timezone');
}
echo "<br>";

/* ----------------------------------------------------------------- */
/*                       Local helper functions                      */

//Function
/**
 * Berechnet das Startdatum, Enddatum und das Datum vom Tag zuvor (gestern).
 *
 * @param DateTime $date Das Basisdatum
 * @param int $startOffset Anzahl der Tage ab heute für das Startdatum
 * @param int $endOffset Anzahl der Tage ab Startdatum für das Enddatum
 * @return array Ein Array mit 'yesterdayDate', 'startDate', 'endDate' und ihren Zeitformaten
 */
function calculateDateRangeWithYesterday(DateTime $date, $startOffset = 0, $endOffset = 0, $debug = false) {
    if ($debug) echo "<strong>Original \$date (vor allem):</strong> " . $date->format('Y-m-d H:i:s') . "<br>";

    // Startdatum
    $start = (clone $date)->add(new DateInterval("P{$startOffset}D"))->setTime(0, 0, 0);
    $rangeStart = $start->format('Y-m-d');
    $rangeStartMitZeit = $start->format('Y-m-d\TH:i:s');
    if ($debug) echo "<strong>Start:</strong> " . $rangeStartMitZeit . "<br>";

    // Yesterday = 1 Tag vor Startdatum
    $yesterday = (clone $start)->sub(new DateInterval("P1D"));
    $rangeStartYesterday = $yesterday->format('Y-m-d');
    $rangeStartYesterdayMitZeit = $yesterday->format('Y-m-d\TH:i:s');
    if ($debug) echo "<strong>Yesterday:</strong> " . $rangeStartYesterdayMitZeit . "<br>";


    // Enddatum
    $end = (clone $date)->add(new DateInterval("P{$endOffset}D"))->setTime(23, 59, 59);
    $rangeEnd = $end->format('Y-m-d');
    $rangeEndMitZeit = $end->format('Y-m-d\TH:i:s');
    if ($debug) echo "<strong>Ende:</strong> " . $rangeEndMitZeit . "<br>";


        // endpluseins = 1 Tag nach EndDatum
        $endPlusOne = (clone $end)->add(new DateInterval("P1D"));
        $rangeEndPlusOne = $endPlusOne->format('Y-m-d');
        $rangeEndPlusOneMitZeit = (clone $endPlusOne)->setTime(23, 59, 59)->format('Y-m-d\TH:i:s');
        if ($debug) echo "<strong>tomorrow:</strong> " . $rangeEndPlusOneMitZeit . "<br>";      
        

    if ($debug) echo "<strong>Finaler \$date:</strong> " . $date->format('Y-m-d H:i:s') . "<br>";

    return [
        'yesterdayDate' => $rangeStartYesterday,
        'yesterdayDateTime' => $rangeStartYesterdayMitZeit,
        'endpluseinsDate' => $rangeEndPlusOne,
        'endpluseinsDateTime' => $rangeEndPlusOneMitZeit,
        'startDate' => $rangeStart,
        'startDateTime' => $rangeStartMitZeit,
        'endDate' => $rangeEnd,
        'endDateTime' => $rangeEndMitZeit,
    ];
}


/**
 * Validiert, ob ein Datum im Format YYYY-MM-DD gültig ist.
 *
 * @param string $date Datum im Format YYYY-MM-DD
 * @return bool True, wenn das Datum gültig ist, sonst False
 */
function isValidDate(string $date): bool {
    $format = 'Y-m-d';
    $dateTime = DateTime::createFromFormat($format, $date);
    return $dateTime && $dateTime->format($format) === $date;
}

// Timer-Hilfsfunktion
function startTimer() {
    return microtime(true);
}

function stopTimerForFunctions($startTime, $taskName) {
    $duration = microtime(true) - $startTime;
    echo "<pre>$taskName dauerte: " . round($duration, 4) . " Sekunden</pre>";
}

function stopTimerForLogs($startTime) {
    $duration = microtime(true) - $startTime;
    return (int)$duration;
}

/* ----------------------------------------------------------------- */
?>