<?php

include_once "log_james.php";
include_once "globals.php";

set_time_limit(0);
date_default_timezone_set('Europe/Berlin');

/* ----------------------------------------------------------------- */
/*                       POST/GET-Parameter verarbeiten              */
// Priorität für POST, Fallback auf GET
$startDate = isset($_POST['start']) ? $_POST['start'] : (isset($_GET['start']) ? $_GET['start'] : 0);
$endDate = isset($_POST['end']) ? $_POST['end'] : (isset($_GET['end']) ? $_GET['end'] : 1);

// Validierung der Datumswerte
if (!$startDate || !$endDate || !isValidDate($startDate) || !isValidDate($endDate)) {
    die("Fehler: Ungültige oder fehlende Start- oder Enddatumswerte. Erwartetes Format: YYYY-MM-DD");
}

if (strtotime($endDate) < strtotime($startDate)) {
    die("Fehler: Das Enddatum muss nach oder gleich dem Startdatum sein.");
}

/* ----------------------------------------------------------------- */
/*                              RangeTime                            */

$timeZone = 'Europe/Berlin';
$date = new DateTime('today', new DateTimeZone($timeZone));
$now = new DateTime('now', new DateTimeZone($timeZone));

// Datenbank Delete Time
$datenbank_vorgestern = clone $date;
$datenbank_vorgestern->sub(new DateInterval("P2D"))->setTime(0, 0, 0);
$datenbank_vorgestern = $datenbank_vorgestern->format('Y-m-d H:i:s');

$datenbank_vor_10_tagen = clone $date;
$datenbank_vor_10_tagen->sub(new DateInterval("P10D"))->setTime(0, 0, 0);
$datenbank_vor_10_tagen = $datenbank_vor_10_tagen->format('Y-m-d H:i:s');

// Datum berechnen
$dateRange = calculateDateRangeWithYesterdayFromStrings($startDate, $endDate);


$rangeStartYesterday = $dateRange['yesterdayDate'];
$rangeStart = $dateRange['startDate'];
$rangeEnd = $dateRange['endDate'];

$rangeStartYesterdayMitZeit = $dateRange['yesterdayDateTime'];
$rangeStartMitZeit = $dateRange['startDateTime'];
$rangeEndMitZeit = $dateRange['endDateTime'];

// Shutdown-Funktion für Fehlerbehandlung
register_shutdown_function(function () {
    $error = error_get_last();
    if ($error) {
        echo "<pre>";
        echo "Das Skript wurde unterbrochen. Letzter Fehler:\n";
        print_r($error);
        echo "</pre>";
    } else {
        echo "Das Skript wurde erfolgreich ausgeführt oder beendet.";
    }
    exit;
});

/* ----------------------------------------------------------------- */
/*                          Beispielverwendung                       */
echo "rangeStart\n";
echo "<br>";
echo $rangeStart;
echo "<br>";
echo "rangeEnd\n";
echo "<br>";
echo $rangeEnd;
echo "<br>";
echo "rangeStartMitZeit\n";
echo "<br>";
echo $rangeStartMitZeit;
echo "<br>";
echo "rangeEndMitZeit\n";
echo "<br>";
echo $rangeEndMitZeit;
echo "<br>";
echo "rangeStartYesterday\n";
echo "<br>";
echo $rangeStartYesterday;
echo "<br>";
echo "rangeStartYesterdayMitZeit\n";
echo "<br>";
echo $rangeStartYesterdayMitZeit;
echo "<br>";
echo "datenbank_vor_10_tagen\n";
echo "<br>";
echo $datenbank_vor_10_tagen;
echo "<br>";



// Zeitzonen-Informationen
echo 'Globale Standardzeitzone: ' . date_default_timezone_get() . "<br>";
echo 'Einstellung in php.ini: ' . ini_get('date.timezone') . "<br>";

if (date_default_timezone_get()) {
    echo 'date_default_timezone_set: ' . date_default_timezone_get() . '<br />';
}

if (ini_get('date.timezone')) {
    echo 'date.timezone: ' . ini_get('date.timezone');
}
echo "<br>";

/* ----------------------------------------------------------------- */
/*                       Local helper functions                      */

/**
 * Berechnet das Startdatum, Enddatum und das Datum vom Vortag (gestern).
 *
 * @param string $startDate Datum im Format YYYY-MM-DD
 * @param string $endDate Datum im Format YYYY-MM-DD
 * @param string $timeZone Zeitzone (Standard: Europe/Berlin)
 * @param bool $debug Debugausgabe
 * @return array Ein Array mit yesterdayDate, startDate, endDate und ihren Zeitformaten
 */
function calculateDateRangeWithYesterdayFromStrings(string $startDate, string $endDate, string $timeZone = 'Europe/Berlin', bool $debug = false): array {
    $tz = new DateTimeZone($timeZone);

    $start = new DateTime($startDate, $tz);
    $end = new DateTime($endDate, $tz);

    // Startzeit setzen
    $start->setTime(0, 0, 0);
    $rangeStart = $start->format('Y-m-d');
    $rangeStartMitZeit = $start->format('Y-m-d\TH:i:s');
    if ($debug) echo "<strong>Start:</strong> $rangeStartMitZeit<br>";

    // Yesterday
    $yesterday = (clone $start)->sub(new DateInterval("P1D"));
    $rangeStartYesterday = $yesterday->format('Y-m-d');
    $rangeStartYesterdayMitZeit = $yesterday->format('Y-m-d\TH:i:s');
    if ($debug) echo "<strong>Yesterday:</strong> $rangeStartYesterdayMitZeit<br>";

    // Endzeit setzen
    $end->setTime(23, 59, 59);
    $rangeEnd = $end->format('Y-m-d');
    $rangeEndMitZeit = $end->format('Y-m-d\TH:i:s');
    if ($debug) echo "<strong>Ende:</strong> $rangeEndMitZeit<br>";

    // Tag nach Ende
    $endPlusOne = (clone $end)->add(new DateInterval("P1D"))->setTime(23, 59, 59);
    $rangeEndPlusOne = $endPlusOne->format('Y-m-d');
    $rangeEndPlusOneMitZeit = $endPlusOne->format('Y-m-d\TH:i:s');
    if ($debug) echo "<strong>morgen (nach Ende):</strong> $rangeEndPlusOneMitZeit<br>";

    return [
        'yesterdayDate' => $rangeStartYesterday,
        'yesterdayDateTime' => $rangeStartYesterdayMitZeit,
        'endpluseinsDate' => $rangeEndPlusOne,
        'endpluseinsDateTime' => $rangeEndPlusOneMitZeit,
        'startDate' => $rangeStart,
        'startDateTime' => $rangeStartMitZeit,
        'endDate' => $rangeEnd,
        'endDateTime' => $rangeEndMitZeit,
    ];
}


/**
 * Validiert, ob ein Datum im Format YYYY-MM-DD gültig ist.
 *
 * @param string $date Datum im Format YYYY-MM-DD
 * @return bool True, wenn das Datum gültig ist, sonst False
 */
function isValidDate(string $date): bool {
    $format = 'Y-m-d';
    $dateTime = DateTime::createFromFormat($format, $date);
    return $dateTime && $dateTime->format($format) === $date;
}

// Timer-Hilfsfunktion
function startTimer() {
    return microtime(true);
}

function stopTimerForFunctions($startTime, $taskName) {
    $duration = microtime(true) - $startTime;
    echo "<pre>$taskName dauerte: " . round($duration, 4) . " Sekunden</pre>";
}

function stopTimerForLogs($startTime) {
    $duration = microtime(true) - $startTime;
    return (int)$duration;
}

/* ----------------------------------------------------------------- */
?>